/////////////////////////////////////////////////////////////////////////////////////////
////Program: DirectX.h
////Description: A nice wrapper class for easy usage of the directx SDK
////By: Ben Gottemoller
////Legal: Freeware, do whatever you want with it
////Link Requirements: dxguid.lib ddraw.lib dsound.lib winmm.lib
/////////////////////////////////////////////////////////////////////////////////////////

#ifndef _DXENGINE_H
#define _DXENGINE_H

////INCLUDES/////////////////////////////////////////////////////////////////////////////
#define WIN32_LEAN_AND_MEAN
#define INITGUID

#include <windows.h>
#include <windowsx.h>
#include <mmsystem.h>
#include <objbase.h>
#include <iostream.h> 
#include <conio.h>
#include <stdlib.h>
#include <malloc.h>
#include <memory.h>
#include <string.h>
#include <stdarg.h>
#include <stdio.h>
#include <math.h>
#include <io.h>
#include <fcntl.h>

#include <ddraw.h> 
#include <dinput.h>
#include <dsound.h> 

#include "2DMathLib.h"
#include "asciitable.h"
/////////////////////////////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////////////////////////////
#define RGB16(r,g,b)	(USHORT)((b%32) + ((g%64) << 6) + ((r%32) << 11))
#define RGB32(a,r,g,b)	(UINT)((b%256) + ((g%256) << 8) + ((r%256) << 16) + ((a%256) << 24))

#define KEY_DOWN(vk_code)	((GetAsyncKeyState(vk_code) & 0x8000) ? 1 : 0)
#define KEY_UP(vk_code)		((GetAsyncKeyState(vk_code) & 0x8000) ? 0 : 1)
/////////////////////////////////////////////////////////////////////////////////////////


int _appstate = 0;

LRESULT CALLBACK WindowProc(HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam)
{
	PAINTSTRUCT	ps;		
	HDC			hdc;

	switch(msg)
	{	
		case WM_ACTIVATEAPP:
		{
			_appstate = (wparam == WA_ACTIVE) || (wparam == WA_CLICKACTIVE);
            return(0);
		} break;

		case WM_CREATE: 
        {
			return(0);
		} break;

		case WM_PAINT:
        {
			hdc = BeginPaint(hwnd,&ps);
            EndPaint(hwnd,&ps);
			return(0);
        } break;

		case WM_DESTROY: 
		{
			PostQuitMessage(0);
			return(0);
		} break;

		default:break;
    }

	return (DefWindowProc(hwnd, msg, wparam, lparam));
} 

class CImage;
/////////////////////////////////////////////////////////////////////////////////////////
/////DXENGINE CLASS DEFINITION///////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////
class DXEngine
{
	friend CImage;
public:
	DXEngine();
	
	//Windows Section//
	int		WinInit(HINSTANCE hinstance, char *win_name, char *win_title);
	int		IsActive(void); //Is the application minimized on restored
	
	//DirectDraw Section//
	int		DDInit(DWORD width, DWORD height, DWORD bpp);
	int		DDAttachPalette(LPPALETTEENTRY color_list);
	int		DDRotatePalette(int start_index, int end_index);
	int		DDFlip(void);
	int		DDLockBack(void);
	int		DDUnlockBack(void);
	int		DDWaitRetrace(void);
	int		DDAttachClipper(LPRECT cliplist, DWORD numrects); 
	int		DDSetDestColorKey(DWORD high_col, DWORD low_col);
	int		DDFillBack(DWORD color);
	int		DDFillRect(int x, int y, int width, int height, DWORD color);
	int		DDGetScreenInfo(int *width, int *height, int *bpp);
	int		DDGetSurface(UCHAR **video_buffer, int *lpitch);
	int		DDGetSurface(USHORT **video_buffer, int *lpitch);
	int		DDGetSurface(UINT **video_buffer, int *lpitch);
	int		DDGetDC(HDC *hdc);
	int		DDReleaseDC(HDC hdc);
	int		DDShutdown(void);

	//Exterior Needs Section//
	int		DDCreatePalette(DWORD dwFlags, LPPALETTEENTRY lpColorTable, LPDIRECTDRAWPALETTE *lplpDDPalette);
	int		DDCreateSurface(LPDDSURFACEDESC2 lpddsd, LPDIRECTDRAWSURFACE7 *lplpdds);
	
	//Common Graphics Section//
	__forceinline int		SetPixel(int x, int y, DWORD color);
	__forceinline DWORD		GetPixel(int x, int y);
	__forceinline int		Line(int x1, int y1, int x2, int y2, DWORD color);
	__forceinline int		HLine(int x1, int x2, int y, DWORD color);
	__forceinline int		Ellipse(int x, int y, int x_radius, int y_radius, DWORD color); 
	__forceinline int		Triangle(int x1, int y1, int x2, int y2, int x3, int y3, DWORD color);
	__forceinline int		FillTriangle(int x1, int y1, int x2, int y2, int x3, int y3, DWORD color);
	__forceinline int		SetText(int x, int y, char *message, DWORD color);

	//Miscellaneous Section//
	void	Error(char *local, char *sublocal);

   ~DXEngine();	

	HWND	  hwnd;

private:
		
	char     *name;
	char     *title;
	HINSTANCE hinst;
	
	enum BufferState
	{
		UNLOCKED,
		LOCKED
	}bbstate;

	INT							Width, Height, Bpp;
	LPRECT						clipperlist;
	LPDIRECTDRAW7               lpdd;
	LPDIRECTDRAWSURFACE7        lpddsprimary;
	LPDIRECTDRAWSURFACE7        lpddsback; 
	LPDIRECTDRAWSURFACE7	    lpddsone;
	LPDIRECTDRAWCLIPPER			lpddclipper;
	LPDIRECTDRAWPALETTE			lpddpal;
	DDSURFACEDESC2				ddsd;
};
/////////////////////////////////////////////////////////////////////////////////////////
/////END OF DXENGINE CLASS DEFINITION////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////////////////////////////
/////DXENGINE CLASS MEMBERS//////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////
DXEngine::DXEngine()
{
	hwnd = 0;
	hinst = 0;
	bbstate = UNLOCKED;
	Width = 0;
	Height = 0;
	Bpp = 0;
	lpdd = NULL;
	lpddsprimary = NULL;
	lpddsback = NULL;
	lpddsone = NULL;
	lpddclipper = NULL;
	lpddpal = NULL;
}


int DXEngine::WinInit(HINSTANCE hinstance, char *win_name, char *win_title)
{
	name = win_name;
	title = win_title;

	WNDCLASS winclass;
	hwnd = 0;
	hinst = 0;

	winclass.style			= CS_DBLCLKS | CS_OWNDC | CS_HREDRAW | CS_VREDRAW;
	winclass.lpfnWndProc	= WindowProc;
	winclass.cbClsExtra		= 0;
	winclass.cbWndExtra		= 0;
	winclass.hInstance		= hinstance;
	winclass.hIcon			= LoadIcon(NULL, IDI_APPLICATION);
	winclass.hCursor		= LoadCursor(NULL, IDC_ARROW);
	winclass.hbrBackground	= (HBRUSH) GetStockObject(BLACK_BRUSH);
	winclass.lpszMenuName	= NULL; 
	winclass.lpszClassName	= name;

	if (!RegisterClass(&winclass)) return(0);

	hwnd = CreateWindow( name, 
                         title,
						 WS_POPUP | WS_MAXIMIZE,
					 	 0,0,	   
						 GetSystemMetrics(SM_CXSCREEN),
						 GetSystemMetrics(SM_CYSCREEN),
						 NULL,	   
						 NULL,	   
						 hinstance,
						 NULL
					   );
	
	if(!hwnd) return(0);

	_appstate = 1;
	return(1);
}


int DXEngine::IsActive(void)
{
	return(_appstate);
}


int DXEngine::DDInit(DWORD width, DWORD height, DWORD bpp)
{
    LPDIRECTDRAW lpdd_tmp = NULL;

	if(FAILED(DirectDrawCreate(NULL, &lpdd_tmp, NULL)))
	{
		Error("DXEngine::DDInit()", "DirectDrawCreate()");
	}

	if(FAILED(lpdd_tmp->QueryInterface(IID_IDirectDraw7, (LPVOID *) & lpdd)))
    {
		Error("DXEngine::DDInit()", "lpdd->QueryInterface()");
	}
    
	if(FAILED(lpdd->SetCooperativeLevel(hwnd, DDSCL_EXCLUSIVE | 
		DDSCL_FULLSCREEN | DDSCL_ALLOWMODEX | DDSCL_ALLOWREBOOT)))
	{
		Error("DXEngine::DDInit()", "lpdd->SetCooperativeLevel()");
	}
	
	if(FAILED(lpdd->SetDisplayMode(width, height, bpp, 0, 0)))
	{
		Error("DXEngine::DDInit()", "lpdd->SetDisplayMode()");
	}

	ZeroMemory(&ddsd, sizeof(ddsd));    
	ddsd.dwSize = sizeof(ddsd);
    ddsd.dwFlags = DDSD_CAPS | DDSD_BACKBUFFERCOUNT;
    ddsd.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE | DDSCAPS_FLIP | DDSCAPS_COMPLEX;
    ddsd.dwBackBufferCount = 1;
    
	if(FAILED(lpdd->CreateSurface(&ddsd, &lpddsprimary, NULL)))
	{
		Error("DXEngine::DDInit()", "lpdd->CreateSurface()");
	}
    
    ddsd.ddsCaps.dwCaps = DDSCAPS_BACKBUFFER;
    
	if(FAILED(lpddsprimary->GetAttachedSurface(&ddsd.ddsCaps, &lpddsback)))
	{
		Error("DXEngine::DDInit()", "lpddsprimary->GetAttachedSurface()");
	}
  
	lpddsprimary->Unlock(NULL);
	lpddsback->Unlock(NULL);

	Width = width;
	Height = height;
	Bpp = bpp;
	return(1);
}


int	DXEngine::DDAttachPalette(LPPALETTEENTRY color_list)
{
	if(Bpp != 8) return 0;

	if(color_list != NULL)
	{
		if(FAILED(lpdd->CreatePalette(DDPCAPS_8BIT | DDPCAPS_ALLOW256 | DDPCAPS_INITIALIZE, color_list, &lpddpal, NULL)))
		{
			Error("DXEngine::DDAttachPalette()", "CreatePalette()");
			return 0;
		}
	}
	else
	{
		PALETTEENTRY palette[256];
		HDC hdc = GetDC(hwnd);
		GetSystemPaletteEntries(hdc, 0, 256, palette);
		
		if(FAILED(lpdd->CreatePalette(DDPCAPS_8BIT | DDPCAPS_ALLOW256 | DDPCAPS_INITIALIZE, palette, &lpddpal, NULL)))
		{
			Error("DXEngine::DDAttachPalette()", "CreatePalette()");
			return 0;
		}
	}

	if(FAILED(lpddsprimary->SetPalette(lpddpal)))
	{
		Error("DXEngine::DDAttachPalette()", "lpddsprimary->SetPalette()");
		return 0;
	}

	return 1;
}


int	DXEngine::DDRotatePalette(int start_index, int end_index)
{
	if(Bpp != 8) return 0;
	if(start_index >= end_index) return 0;
	if(start_index < 0) start_index = 0;
	if(end_index > 255) end_index = 255;

	PALETTEENTRY palette[256];
	int count = end_index - start_index + 1;
	
	lpddpal->GetEntries(0, start_index, count, palette);
	lpddpal->SetEntries(0, start_index + 1, count - 1, palette);
	lpddpal->SetEntries(0, start_index, 1, &palette[count - 1]);

	return 1;
}


int DXEngine::DDFlip(void)
{
	DDUnlockBack();
	
	while (FAILED(lpddsprimary->Flip(NULL, 0)))
    {
		if(lpddsprimary->IsLost()) lpddsprimary->Restore();
        if(lpddsback->IsLost()) lpddsback->Restore();
    }
	return 1;
}


int DXEngine::DDLockBack(void)
{
	if(bbstate == LOCKED) return 1;

	ZeroMemory(&ddsd, sizeof(ddsd));
	ddsd.dwSize = sizeof(ddsd);
	while (lpddsback->Lock(NULL, &ddsd, 0, NULL) == DDERR_WASSTILLDRAWING);
	bbstate = LOCKED;	
	return 1;
}


int DXEngine::DDUnlockBack(void)
{
	if(bbstate == UNLOCKED) return 1;

	if(FAILED(lpddsback->Unlock(NULL)))
	{
		Error("DDUnlockBack()", "lpddsback->Unlock()");
		return 0;
	}

	bbstate = UNLOCKED;
	return 1;
}


int DXEngine::DDWaitRetrace(void)
{
	lpdd->WaitForVerticalBlank(DDWAITVB_BLOCKBEGIN,0);
	return 1;
}


int DXEngine::DDAttachClipper(LPRECT clip_list, DWORD num_rects)
{
	DWORD index;
	LPRGNDATA region_data;

	if(FAILED(lpdd->CreateClipper(0, &lpddclipper, NULL)))
	{
		Error("DDAttachClipper()", "lpdd->CreateClipper()");
		return 0;
	}

	region_data = (LPRGNDATA)malloc(sizeof(RGNDATAHEADER) + num_rects * sizeof(RECT));
	memcpy(region_data->Buffer, clip_list, sizeof(RECT) * num_rects);

	region_data->rdh.dwSize			= sizeof(RGNDATAHEADER);
	region_data->rdh.iType			= RDH_RECTANGLES;
	region_data->rdh.nCount			= num_rects;
	region_data->rdh.nRgnSize		= num_rects * sizeof(RECT);
	
	region_data->rdh.rcBound.left	=  64000;	
	region_data->rdh.rcBound.top	=  64000;
	region_data->rdh.rcBound.right	= -64000;
	region_data->rdh.rcBound.bottom	= -64000;

	for(index=0; index<num_rects; index++)
	{
		if(clip_list[index].left < region_data->rdh.rcBound.left)
		{
			region_data->rdh.rcBound.left = clip_list[index].left;
		}

		if(clip_list[index].right > region_data->rdh.rcBound.right)
		{
			region_data->rdh.rcBound.right = clip_list[index].right;
		}

		if(clip_list[index].top < region_data->rdh.rcBound.top)
		{
			region_data->rdh.rcBound.top = clip_list[index].top;
		}

		if(clip_list[index].bottom > region_data->rdh.rcBound.bottom)
		{
			region_data->rdh.rcBound.bottom = clip_list[index].bottom;
		}
	}

	if(FAILED(lpddclipper->SetClipList(region_data, 0)))
	{
		free(region_data);
		Error("DDAttachClipper()", "lpddclipper->SetClipList()");
		return 0;
	}

	if(FAILED(lpddsback->SetClipper(lpddclipper)))
	{
		free(region_data);
		Error("DDAttachClipper()", "lpddsback->SetClipper()");
		return 0;
	}

	free(region_data);
	return 1;
}


int	DXEngine::DDSetDestColorKey(DWORD high_col, DWORD low_col)
{
	DDCOLORKEY ddck;
	ddck.dwColorSpaceLowValue = low_col;
	ddck.dwColorSpaceHighValue = high_col;
	if(FAILED(lpddsback->SetColorKey(DDCKEY_DESTBLT | DDCKEY_COLORSPACE, &ddck)))
	{
		return 0;
	}
	return 1;
}	


int DXEngine::DDFillBack(DWORD color)
{
	DDBLTFX ddbltfx; 

	ZeroMemory(&ddbltfx, sizeof(ddbltfx));
	ddbltfx.dwSize = sizeof(ddbltfx);
	ddbltfx.dwFillColor = color; 

	if(lpddsprimary->IsLost()) lpddsprimary->Restore();
    if(lpddsback->IsLost()) lpddsback->Restore();
	
	if(FAILED(lpddsback->Blt(NULL, NULL, NULL, DDBLT_COLORFILL | DDBLT_WAIT, &ddbltfx)))
	{
		Error("DDFillBack()", "lpddsback->Blt()");
		return 0;
	}
	
	return(1);
}


int DXEngine::DDFillRect(int x, int y, int width, int height, DWORD color)
{
	DDBLTFX ddbltfx;
	RECT rect;
	
	ZeroMemory(&ddbltfx, sizeof(ddbltfx));
	ddbltfx.dwSize = sizeof(ddbltfx);
	ddbltfx.dwFillColor = color;

	rect.left = x;
	rect.top = y;
	rect.right = x + width;
	rect.bottom = y + height;

	if(lpddsprimary->IsLost()) lpddsprimary->Restore();
    if(lpddsback->IsLost()) lpddsback->Restore();

	if(FAILED(lpddsback->Blt(&rect, NULL, NULL, DDBLT_COLORFILL | DDBLT_WAIT, &ddbltfx)))
	{
		Error("DDFillRect()", "lpddsback->Blt()");
		return 0;
	}
	return 1;
}


int DXEngine::DDGetScreenInfo(int *width, int *height, int *bpp)
{
	if(width != NULL) *width = Width;
	if(height != NULL) *height = Height;
	if(bpp != NULL) *bpp = Bpp;
	return 1;
}


int DXEngine::DDGetSurface(UCHAR **video_buffer, int *lpitch)
{
	*video_buffer = (UCHAR *)ddsd.lpSurface;
	*lpitch = (int)ddsd.lPitch;
	return 1;
}

int DXEngine::DDGetSurface(USHORT **video_buffer, int *lpitch)
{
	*video_buffer = (USHORT *)ddsd.lpSurface;
	*lpitch = (int)(ddsd.lPitch >> 1);
	return 1;
}

int DXEngine::DDGetSurface(UINT **video_buffer, int *lpitch)
{
	*video_buffer = (UINT *)ddsd.lpSurface;
	*lpitch = (int)(ddsd.lPitch >> 2);
	return 1;
}


int DXEngine::DDGetDC(HDC *hdc)
{
	lpddsback->GetDC(hdc);
	return 1;
}


int DXEngine::DDReleaseDC(HDC hdc)
{
	lpddsback->ReleaseDC(hdc);
	return 1;
}


int DXEngine::DDShutdown(void)
{
	if(lpdd != NULL)
	{
		if(lpddsone != NULL)
		{
			lpddsone->Unlock(NULL);
			lpddsone->Release();
			lpddsone = NULL;
		}

		if(lpddsback != NULL)
		{
			lpddsback->Unlock(NULL);
			lpddsback->Release();
			lpddsback = NULL;
		}

		if(lpddsprimary != NULL)
		{
			lpddsprimary->Unlock(NULL);
			lpddsprimary->Release();
			lpddsprimary = NULL;
		}

		lpdd->Release();
		lpdd = NULL;
	}		

	_appstate = 0;
	return 1;
}


int	DXEngine::DDCreatePalette(DWORD dwFlags, LPPALETTEENTRY lpColorTable, LPDIRECTDRAWPALETTE *lplpDDPalette)
{
	if(FAILED(lpdd->CreatePalette(dwFlags, lpColorTable, lplpDDPalette, NULL))) return 0;
	return 1;
}


int	DXEngine::DDCreateSurface(LPDDSURFACEDESC2 lpddsd, LPDIRECTDRAWSURFACE7 *lplpdds)
{
	if(FAILED(lpdd->CreateSurface(lpddsd, lplpdds, NULL))) return 0;
	return 1;
}


__forceinline int DXEngine::SetPixel(int x, int y, DWORD color)
{
	if(bbstate == UNLOCKED) return 0;
	if((x < 0) || (x > (Width-1)) || (y < 0) || (y > (Height-1))) return 0;
	switch(Bpp)
	{
		case 8:
		{
			((UCHAR *)ddsd.lpSurface)[x + (y * (int)ddsd.lPitch)] = (UCHAR)color;
		} break;
		
		case 16:
		{
			((USHORT *)ddsd.lpSurface)[x + (y * (int)ddsd.lPitch >> 1)] = (USHORT)color;
		} break;

		case 32:
		{
			((UINT *)ddsd.lpSurface)[x + (y * (int)ddsd.lPitch >> 2)] = (UINT)color;
		} break;
	}
	return 1;
}


__forceinline DWORD DXEngine::GetPixel(int x, int y)
{
	if(bbstate == UNLOCKED) return 0;
	if((x < 0) || (x > (Width-1)) || (y < 0) || (y > (Height-1))) return 0;
	switch(Bpp)
	{
		case 8:
		{
			return (UCHAR)((UCHAR *)ddsd.lpSurface)[x + (y * (int)ddsd.lPitch)];
		} break;
		
		case 16:
		{
			return (USHORT)((USHORT *)ddsd.lpSurface)[x + (y * (int)ddsd.lPitch >> 1)];
		} break;

		case 32:
		{
			return (UINT)((UINT *)ddsd.lpSurface)[x + (y * (int)ddsd.lPitch >> 2)];
		} break;
	}
	return 0;
}


__forceinline int DXEngine::Line(int x1, int y1, int x2, int y2, DWORD color)
{
	if(bbstate == UNLOCKED) return 0;
	
	int x_unit = 1;
	int y_unit = 1;
	int x_diff = x2 - x1;
	int y_diff = y2 - y1;
	int error_term = 0;

//////TESTING////////////////////////////////////////////////
/*	if(x1 < 0) 
	{
		y1 += ((-x1) * y_diff) / x_diff;	//Good:)
		x1 = 0;
	}
	if(x1 > Width) 
	{
		y1 += ((-x1) * y_diff) / x_diff;	
		x1 = Width;
	}	
	if(y1 < 0) 
	{
		x1 += ((-y1) * x_diff) / y_diff; 
		y1 = 0;
	}
	if(y1 > Height)
	{
		x1 += ((-y1) * x_diff) / y_diff;
		y1 = Height;
	}
	
	if(x2 < 0) 
	{
		y2 += ((-x2) * y_diff) / x_diff;	
		x2 = 0;
	}
	if(x2 > Width) 
	{
		y2 += ((-x2) * y_diff) / x_diff;	
		x2 = Width;
	}	
	if(y2 < 0) 
	{
		x2 += ((-y2) * x_diff) / y_diff; 
		y2 = 0;
	}
	if(y2 > Height)
	{
		x2 += ((-y2) * x_diff) / y_diff;
		y2 = Height;
	}
*/
////////////////////////////////////////////////////////////

	if(x_diff < 0)
	{
		x_diff = -x_diff;
		x_unit = -1;
	}

	if(y_diff < 0)
	{
		y_diff = -y_diff;
		y_unit = -1;
	}

	if(x_diff > y_diff)
	{
		int length = x_diff + 1;
		for(int i=0; i<length; i++)
		{
			SetPixel(x1, y1, color);
			x1 += x_unit;
			error_term += y_diff;
			if(error_term > x_diff)
			{
				error_term -= x_diff;
				y1 += y_unit;
			}
		}
	}
	else
	{
		int length = y_diff + 1;
		for(int i=0; i<length; i++)
		{
			SetPixel(x1, y1, color);
			y1 += y_unit;
			error_term += x_diff;
			if(error_term > 0)
			{
				error_term -= y_diff;
				x1 += x_unit;
			}
		}
	}

	return 1;
}


__forceinline int DXEngine::HLine(int x1, int x2, int y, DWORD color)
{
	if(bbstate == UNLOCKED) return 0;
	if((y < 0) || (y >= Height)) return 0;
	if((x1 < 0) && (x2 < 0)) return 0;
	if((x1 >= Width) && (x2 >= Width)) return 0;

	if(x1 > x2) SWAP(x1, x2);
	if(x1 < 0) x1 = 0;
	if(x2 < 0) x2 = 0;
	if(x1 >= Width) x1 = Width - 1;
	if(x2 >= Width) x2 = Width - 1;

	switch(Bpp)
	{
		case 8:
		{
			memset(((UCHAR *)ddsd.lpSurface) + x1 + (y * (int)ddsd.lPitch), 
				   (UCHAR)color, sizeof(UCHAR) * (x2 - x1));
		} break;
		
		case 16:
		{
			for(int i=x1; i<x2; i++)
			{
				((USHORT *)ddsd.lpSurface)[i + (y * (int)ddsd.lPitch >> 1)] = (USHORT)color;
			}
		} break;

		case 32:
		{
			for(int i=x1; i<x2; i++)
			{
				((UINT *)ddsd.lpSurface)[i + (y * (int)ddsd.lPitch >> 2)] = (UINT)color;
			}
		} break;
	}
	return 1;
}


__forceinline int DXEngine::Ellipse(int x, int y, int x_radius, int y_radius, DWORD color)
{
	if(bbstate == UNLOCKED) return 0;

	int num_points = 60;
	int pt_unit = 256 / num_points;
	int x1 = x_radius, y1 = 0, 
		x2 = 0, y2 = 0;

	for(int degree=pt_unit; degree<256; degree+=pt_unit)
	{
		x2 = (int)(x_radius * COS256(degree));
		y2 = (int)(y_radius * SIN256(degree));
		Line(x + x1, y + y1, x + x2, y + y2, color);
		x1 = x2;
		y1 = y2;
	}
	
	Line(x + x1, y + y1, x + x_radius, y, color);

	return 1;
}


__forceinline int	DXEngine::Triangle(int x1, int y1, int x2, int y2, int x3, int y3, DWORD color)
{
	if(bbstate == UNLOCKED) return 0;
	if(((x1 == x2) && (x2 == x3)) || ((y1 == y2) && (y2 == y3))) return 0;

	Line(x1, y1, x2, y2, color);
	Line(x1, y1, x3, y3, color);
	Line(x2, y2, x3, y3, color);
	
	return 1;
}


__forceinline int DXEngine::FillTriangle(int x1, int y1, int x2, int y2, int x3, int y3, DWORD color)
{
	if(bbstate == UNLOCKED) return 0;
	if(((x1 == x2) && (x2 == x3)) || ((y1 == y2) && (y2 == y3))) return 0;

	if(y1 > y2) {SWAP(x1,x2); SWAP(y1,y2);}
	if(y2 > y3) {SWAP(x2,x3); SWAP(y2,y3);}
	if(y1 > y2) {SWAP(x1,x2); SWAP(y1,y2);}
	
	float x_left, x_right;
	float lslope, rslope;
	int height;

	if(y1 == y2) //flat top tri
	{
		if(x1 > x2) {SWAP(x1,x2); SWAP(y1,y2);}
		
		x_left = (float)x1;
		x_right = (float)x2;
		lslope = (float)(x3 - x1) / (float)(y3 - y1);
		rslope = (float)(x3 - x2) / (float)(y3 - y2);
		
		for(height=y1; height<y3; height++)
		{
			HLine((int)(x_left+0.5), (int)(x_right+0.5), height, color);
			x_left += lslope;
			x_right += rslope;
		}
	}
	else
	if(y2 == y3) //flat bottom tri
	{
		if(x2 > x3) {SWAP(x2,x3); SWAP(y2,y3);}
		
		x_left = (float)x1;
		x_right = (float)x1;

		lslope = (float)(x2 - x1) / (float)(y2 - y1);
		rslope = (float)(x3 - x1) / (float)(y3 - y1);
		
		for(height = y1; height<y2; height++)
		{
			HLine((int)(x_left+0.5), (int)(x_right+0.5), height, color);
			x_left += lslope;
			x_right += rslope;
		}
	}
	else //break up tri into flat top & flat bottom
	{
		x_left = (float)x1;
		x_right = (float)x1;

		if(x2 < x3)
		{
			lslope = (float)(x2 - x1) / (float)(y2 - y1);
			rslope = (float)(x3 - x1) / (float)(y3 - y1);
		}
		else
		{
			lslope = (float)(x3 - x1) / (float)(y3 - y1);
			rslope = (float)(x2 - x1) / (float)(y2 - y1);
		}

		for(height=y1; height<y2; height++)
		{
			HLine((int)(x_left+0.5), (int)(x_right+0.5), height, color);
			x_left += lslope;
			x_right += rslope;
		}
	
		if(x2 < x3) 
		{
			lslope = (float)(x3 - x2) / (float)(y3 - y2);
		}
		else
		{
			rslope = (float)(x3 - x2) / (float)(y3 - y2);
		}

		for(height=y2; height<y3; height++)
		{
			HLine((int)(x_left+0.5), (int)(x_right+0.5), height, color);
			x_left += lslope;
			x_right += rslope;
		}
	}

	return 1;
}


__forceinline int DXEngine::SetText(int x, int y, char *message, DWORD color)
{
	int spacing = 16;
	if(bbstate == UNLOCKED) return 0;

	int ascii_index = 0;
	int length = strlen(message);
	int xp, yp;
	
	for(int index=0; index<length; index++)
	{
		for(yp=0; yp<16; yp++)
		{
			for(xp=0; xp<16; xp++)
			{
				ascii_index = message[index] - 32;
				if((ascii_index < 0) || (ascii_index >= 96)) ascii_index = 0;

				if(ASCII_TABLE[ascii_index].letter[xp + (yp * 16)])
				{
					SetPixel(x+xp+(index*spacing), y+yp, color);
				}
			}
		}
	}
	return 1;
}


void DXEngine::Error(char *local, char *sublocal)
{
	char message[200];
	ShowCursor(1);
	sprintf(message, "Location: %s\nSubLocal: %s", local, sublocal);
	DDShutdown();
	MessageBox(hwnd, message, "ERROR!!!", MB_OK | MB_ICONEXCLAMATION);
	PostMessage(hwnd, WM_DESTROY,0,0);
}


DXEngine::~DXEngine()
{
	if(lpddsone != NULL)
	{
		lpddsone->Unlock(NULL);
		lpddsone->Release();
		lpddsone = NULL;
	}

	if(lpddsback)
	{
		lpddsback->Unlock(NULL);
		lpddsback->Release();
		lpddsback = NULL;
	}

	if(lpddsprimary)
	{
		lpddsprimary->Unlock(NULL);
		lpddsprimary->Release();
		lpddsprimary = NULL;
	}

	if(lpdd)
	{
		lpdd->Release();
		lpdd = NULL;
	}

	_appstate = 0;
}
/////////////////////////////////////////////////////////////////////////////////////////
/////END OF DXENGINE CLASS MEMBERS///////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////

#endif